/*

   See the NOTICE file distributed with this work for additional
   information regarding copyright ownership.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */
package io.sf.carte.echosvg.anim.dom;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.svg.SVGAnimatedString;
import org.w3c.dom.svg.SVGGlyphRefElement;

import io.sf.carte.echosvg.constants.XMLConstants;
import io.sf.carte.echosvg.dom.AbstractDocument;

/**
 * This class implements {@link SVGGlyphRefElement}.
 *
 * @author <a href="mailto:stephane@hillion.org">Stephane Hillion</a>
 * @author For later modifications, see Git history.
 * @version $Id$
 */
public class SVGOMGlyphRefElement extends SVGStylableElement implements SVGGlyphRefElement {

//     /**
//      * Table mapping XML attribute names to TraitInformation objects.
//      */
//     protected static DoublyIndexedTable<String,String> xmlTraitInformation;
//     static {
//         DoublyIndexedTable<String,String> t = new DoublyIndexedTable<>(SVGStylableElement.xmlTraitInformation);
//         t.put(XLINK_NAMESPACE_URI, XLINK_HREF_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_URI));
//         t.put(null, SVG_GLYPH_REF_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_CDATA));
//         t.put(null, SVG_FORMAT_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_CDATA));
//         t.put(null, SVG_X_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_NUMBER));
//         t.put(null, SVG_Y_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_NUMBER));
//         t.put(null, SVG_DX_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_NUMBER));
//         t.put(null, SVG_DY_ATTRIBUTE,
//                 new TraitInformation(false, SVGTypes.TYPE_NUMBER));
//         xmlTraitInformation = t;
//     }

	private static final long serialVersionUID = 2L;

	/**
	 * The attribute initializer.
	 */
	protected static final AttributeInitializer attributeInitializer;
	static {
		attributeInitializer = new AttributeInitializer(4);
		attributeInitializer.addAttribute(XMLConstants.XMLNS_NAMESPACE_URI, null, "xmlns:xlink",
				XMLConstants.XLINK_NAMESPACE_URI);
		attributeInitializer.addAttribute(XMLConstants.XLINK_NAMESPACE_URI, "xlink", "type", "simple");
		attributeInitializer.addAttribute(XMLConstants.XLINK_NAMESPACE_URI, "xlink", "show", "other");
		attributeInitializer.addAttribute(XMLConstants.XLINK_NAMESPACE_URI, "xlink", "actuate", "onLoad");
	}

	/**
	 * The namespaceless 'href' attribute value.
	 */
	private SVGOMAnimatedString href;

	/**
	 * The 'xlink:href' attribute value. Note that this attribute not actually
	 * animatable, according to SVG 1.1.
	 */
	private SVGOMAnimatedString xlinkhref;

	/**
	 * Creates a new SVGOMGlyphRefElement object.
	 */
	protected SVGOMGlyphRefElement() {
	}

	/**
	 * Creates a new SVGOMGlyphRefElement object.
	 * 
	 * @param prefix The namespace prefix.
	 * @param owner  The owner document.
	 */
	public SVGOMGlyphRefElement(String prefix, AbstractDocument owner) {
		super(prefix, owner);
		initializeLiveAttributes();
	}

	/**
	 * Initializes all live attributes for this element.
	 */
	@Override
	protected void initializeAllLiveAttributes() {
		super.initializeAllLiveAttributes();
		initializeLiveAttributes();
	}

	/**
	 * Initializes the live attribute values of this element.
	 */
	private void initializeLiveAttributes() {
		href = createLiveAnimatedString(null, XLINK_HREF_ATTRIBUTE);
		xlinkhref = createLiveAnimatedString(XLINK_NAMESPACE_URI, XLINK_HREF_ATTRIBUTE);
	}

	/**
	 * <b>DOM</b>: Implements {@link Node#getLocalName()}.
	 */
	@Override
	public String getLocalName() {
		return SVG_GLYPH_REF_TAG;
	}

	/**
	 * <b>DOM</b>: Implements {@link org.w3c.dom.svg.SVGURIReference#getHref()}.
	 */
	@Override
	public SVGAnimatedString getHref() {
		return hasAttribute(XLINK_HREF_ATTRIBUTE) ? href : xlinkhref;
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getGlyphRef()}.
	 */
	@Override
	public String getGlyphRef() {
		return getAttributeNS(null, SVG_GLYPH_REF_ATTRIBUTE);
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setGlyphRef(String)}.
	 */
	@Override
	public void setGlyphRef(String glyphRef) throws DOMException {
		setAttributeNS(null, SVG_GLYPH_REF_ATTRIBUTE, glyphRef);
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getFormat()}.
	 */
	@Override
	public String getFormat() {
		return getAttributeNS(null, SVG_FORMAT_ATTRIBUTE);
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setFormat(String)}.
	 */
	@Override
	public void setFormat(String format) throws DOMException {
		setAttributeNS(null, SVG_FORMAT_ATTRIBUTE, format);
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getX()}.
	 */
	@Override
	public float getX() {
		return Float.parseFloat(getAttributeNS(null, SVG_X_ATTRIBUTE));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setX(float)}.
	 */
	@Override
	public void setX(float x) throws DOMException {
		setAttributeNS(null, SVG_X_ATTRIBUTE, String.valueOf(x));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getY()}.
	 */
	@Override
	public float getY() {
		return Float.parseFloat(getAttributeNS(null, SVG_Y_ATTRIBUTE));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setY(float)}.
	 */
	@Override
	public void setY(float y) throws DOMException {
		setAttributeNS(null, SVG_Y_ATTRIBUTE, String.valueOf(y));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getDx()}.
	 */
	@Override
	public float getDx() {
		return Float.parseFloat(getAttributeNS(null, SVG_DX_ATTRIBUTE));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setDx(float)}.
	 */
	@Override
	public void setDx(float dx) throws DOMException {
		setAttributeNS(null, SVG_DX_ATTRIBUTE, String.valueOf(dx));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#getDy()}.
	 */
	@Override
	public float getDy() {
		return Float.parseFloat(getAttributeNS(null, SVG_DY_ATTRIBUTE));
	}

	/**
	 * <b>DOM</b>: Implements {@link SVGGlyphRefElement#setDy(float)}.
	 */
	@Override
	public void setDy(float dy) throws DOMException {
		setAttributeNS(null, SVG_DY_ATTRIBUTE, String.valueOf(dy));
	}

	/**
	 * Returns the AttributeInitializer for this element type.
	 * 
	 * @return null if this element has no attribute with a default value.
	 */
	@Override
	protected AttributeInitializer getAttributeInitializer() {
		return attributeInitializer;
	}

	/**
	 * Returns a new uninitialized instance of this object's class.
	 */
	@Override
	protected Node newNode() {
		return new SVGOMGlyphRefElement();
	}

//     /**
//      * Returns the table of TraitInformation objects for this element.
//      */
//     protected DoublyIndexedTable<String,String> getTraitInformationTable() {
//         return xmlTraitInformation;
//     }
}
