/*

   See the NOTICE file distributed with this work for additional
   information regarding copyright ownership.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */
package io.sf.carte.echosvg.css.engine.value.css2;

import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;

import io.sf.carte.doc.style.css.nsac.LexicalUnit;
import io.sf.carte.echosvg.css.engine.CSSEngine;
import io.sf.carte.echosvg.css.engine.value.IdentifierManager;
import io.sf.carte.echosvg.css.engine.value.ListValue;
import io.sf.carte.echosvg.css.engine.value.StringMap;
import io.sf.carte.echosvg.css.engine.value.StringValue;
import io.sf.carte.echosvg.css.engine.value.URIValue;
import io.sf.carte.echosvg.css.engine.value.Value;
import io.sf.carte.echosvg.css.engine.value.ValueConstants;
import io.sf.carte.echosvg.css.engine.value.ValueManager;
import io.sf.carte.echosvg.util.CSSConstants;
import io.sf.carte.echosvg.util.SVGTypes;

/**
 * One line Class Desc
 *
 * Complete Class Desc
 *
 * @author <a href="mailto:deweese@apache.org">l449433</a>
 * @author For later modifications, see Git history.
 * @version $Id$
 */
public class SrcManager extends IdentifierManager {

	/**
	 * The identifier values.
	 */
	protected static final StringMap values = new StringMap();
	static {
		values.put(CSSConstants.CSS_NONE_VALUE, ValueConstants.NONE_VALUE);
	}

	public SrcManager() {
	}

	/**
	 * Implements
	 * {@link io.sf.carte.echosvg.css.engine.value.ValueManager#isInheritedProperty()}.
	 */
	@Override
	public boolean isInheritedProperty() {
		return false;
	}

	/**
	 * Implements {@link ValueManager#isAnimatableProperty()}.
	 */
	@Override
	public boolean isAnimatableProperty() {
		return false;
	}

	/**
	 * Implements {@link ValueManager#isAdditiveProperty()}.
	 */
	@Override
	public boolean isAdditiveProperty() {
		return false;
	}

	/**
	 * Implements {@link ValueManager#getPropertyType()}.
	 */
	@Override
	public int getPropertyType() {
		return SVGTypes.TYPE_FONT_DESCRIPTOR_SRC_VALUE;
	}

	/**
	 * Implements
	 * {@link io.sf.carte.echosvg.css.engine.value.ValueManager#getPropertyName()}.
	 */
	@Override
	public String getPropertyName() {
		return CSSConstants.CSS_SRC_PROPERTY;
	}

	/**
	 * Implements
	 * {@link io.sf.carte.echosvg.css.engine.value.ValueManager#getDefaultValue()}.
	 */
	@Override
	public Value getDefaultValue() {
		return ValueConstants.NONE_VALUE;
	}

	/**
	 * Implements {@link ValueManager#createValue(LexicalUnit,CSSEngine)}.
	 */
	@Override
	public Value createValue(LexicalUnit lu, CSSEngine engine) throws DOMException {

		switch (lu.getLexicalUnitType()) {
		case INHERIT:
			return ValueConstants.INHERIT_VALUE;

		default:
			throw createInvalidLexicalUnitDOMException(lu.getLexicalUnitType());

		case IDENT:
		case STRING:
		case URI:
		}

		ListValue result = new ListValue();
		for (;;) {
			switch (lu.getLexicalUnitType()) {
			case STRING:
				result.append(new StringValue(CSSPrimitiveValue.CSS_STRING, lu.getStringValue()));
				lu = lu.getNextLexicalUnit();
				break;

			case URI:
				String uri = resolveURI(engine.getCSSBaseURI(), lu.getStringValue());

				result.append(new URIValue(lu.getStringValue(), uri));
				lu = lu.getNextLexicalUnit();
				if ((lu != null) && (lu.getLexicalUnitType() == LexicalUnit.LexicalType.FUNCTION)) {
					if (!lu.getFunctionName().equalsIgnoreCase("format")) {
						break;
					}
					// Format really does us no good so just ignore it.

					// TODO: Should probably turn this into a ListValue
					// and append the format function CSS Value.
					lu = lu.getNextLexicalUnit();
				}
				break;

			case IDENT:
				StringBuilder sb = new StringBuilder(lu.getStringValue());
				lu = lu.getNextLexicalUnit();
				if (lu != null && lu.getLexicalUnitType() == LexicalUnit.LexicalType.IDENT) {
					do {
						sb.append(' ');
						sb.append(lu.getStringValue());
						lu = lu.getNextLexicalUnit();
					} while (lu != null && lu.getLexicalUnitType() == LexicalUnit.LexicalType.IDENT);
					result.append(new StringValue(CSSPrimitiveValue.CSS_STRING, sb.toString()));
				} else {
					String id = sb.toString();
					String s = id.toLowerCase().intern();
					Value v = (Value) values.get(s);
					result.append((v != null) ? v : new StringValue(CSSPrimitiveValue.CSS_STRING, id));
				}
				break;

			default:
				break;

			}
			if (lu == null) {
				return result;
			}
			if (lu.getLexicalUnitType() != LexicalUnit.LexicalType.OPERATOR_COMMA) {
				throw createInvalidLexicalUnitDOMException(lu.getLexicalUnitType());
			}
			lu = lu.getNextLexicalUnit();
			if (lu == null) {
				throw createMalformedLexicalUnitDOMException();
			}
		}
	}

	/**
	 * Implements {@link IdentifierManager#getIdentifiers()}.
	 */
	@Override
	public StringMap getIdentifiers() {
		return values;
	}

}
