/*

   See the NOTICE file distributed with this work for additional
   information regarding copyright ownership.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */
package io.sf.carte.echosvg.css.dom;

import org.w3c.css.om.CSSRule;
import org.w3c.dom.DOMException;

import io.sf.carte.echosvg.css.engine.CSSEngine;
import io.sf.carte.echosvg.css.engine.SVGCSSEngine;
import io.sf.carte.echosvg.css.engine.value.AbstractValueModificationHandler;
import io.sf.carte.echosvg.css.engine.value.CSSVal;
import io.sf.carte.echosvg.css.engine.value.Value;
import io.sf.carte.echosvg.css.engine.value.svg.SVGColorManager;
import io.sf.carte.echosvg.css.engine.value.svg.SVGPaintManager;

/**
 * This class represents a SVG style declaration.
 *
 * <p>
 * Original author: <a href="mailto:stephane@hillion.org">Stephane Hillion</a>.
 * For later modifications, see Git history.
 * </p>
 * @version $Id$
 */
public class CSSOMSVGStyleDeclaration extends CSSOMStyleDeclaration {

	/**
	 * The CSS engine.
	 */
	protected CSSEngine cssEngine;

	/**
	 * Creates a new CSSOMSVGStyleDeclaration.
	 */
	public CSSOMSVGStyleDeclaration(ValueProvider vp, CSSRule parent, CSSEngine eng) {
		super(vp, parent);
		cssEngine = eng;
	}

	/**
	 * Creates the CSS value associated with the given property.
	 */
	@Override
	protected CSSVal createCSSValue(String name) {
		int idx = cssEngine.getPropertyIndex(name);
		if (idx > SVGCSSEngine.FINAL_INDEX) {
			if (cssEngine.getValueManagers()[idx] instanceof SVGPaintManager) {
				return new StyleDeclarationPaintValue(name);
			}
			if (cssEngine.getValueManagers()[idx] instanceof SVGColorManager) {
				return new StyleDeclarationColorValue(name);
			}
		} else {
			switch (idx) {
			case SVGCSSEngine.FILL_INDEX:
			case SVGCSSEngine.STROKE_INDEX:
				return new StyleDeclarationPaintValue(name);

			case SVGCSSEngine.FLOOD_COLOR_INDEX:
			case SVGCSSEngine.LIGHTING_COLOR_INDEX:
			case SVGCSSEngine.STOP_COLOR_INDEX:
				return new StyleDeclarationColorValue(name);
			}
		}
		return super.createCSSValue(name);
	}

	/**
	 * This class represents a CSS value returned by this declaration.
	 */
	public class StyleDeclarationColorValue extends CSSOMSVGColor implements CSSOMSVGColor.ValueProvider {

		/**
		 * The property name.
		 */
		protected String property;

		/**
		 * Creates a new StyleDeclarationColorValue.
		 */
		public StyleDeclarationColorValue(String prop) {
			super(null);
			valueProvider = this;
			setModificationHandler(new AbstractValueModificationHandler() {
				@Override
				protected Value getValue() {
					return StyleDeclarationColorValue.this.getValue();
				}

				@Override
				public void setPropertyText(String text) throws DOMException {
					if (handler == null) {
						throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
					}
					String prio = getPropertyPriority(property);
					CSSOMSVGStyleDeclaration.this.handler.propertyChanged(property, text, prio);
				}
			});

			property = prop;
		}

		// ValueProvider ///////////////////////////////

		/**
		 * Returns the current value associated with this object.
		 */
		@Override
		public Value getValue() {
			return CSSOMSVGStyleDeclaration.this.valueProvider.getValue(property);
		}

	}

	/**
	 * This class represents a CSS value returned by this declaration.
	 */
	public class StyleDeclarationPaintValue extends CSSOMSVGPaint implements CSSOMSVGPaint.ValueProvider {

		/**
		 * The property name.
		 */
		protected String property;

		/**
		 * Creates a new StyleDeclarationPaintValue.
		 */
		public StyleDeclarationPaintValue(String prop) {
			super(null);
			valueProvider = this;
			setModificationHandler(new AbstractModificationHandler() {
				@Override
				protected Value getValue() {
					return StyleDeclarationPaintValue.this.getValue();
				}

				@Override
				public void setPropertyText(String text) throws DOMException {
					if (handler == null) {
						throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
					}
					String prio = getPropertyPriority(property);
					CSSOMSVGStyleDeclaration.this.handler.propertyChanged(property, text, prio);
				}
			});

			property = prop;
		}

		// ValueProvider ///////////////////////////////

		/**
		 * Returns the current value associated with this object.
		 */
		@Override
		public Value getValue() {
			return CSSOMSVGStyleDeclaration.this.valueProvider.getValue(property);
		}

	}

}
